<?php
//Template.class.php
/**
 * Holds the geoTemplate class.
 * 
 * @package System
 * @since Version 4.0.0
 */
/**************************************************************************
Geodesic Classifieds & Auctions Platform 5.2
Copyright (c) 2001-2011 Geodesic Solutions, LLC
All rights reserved
http://geodesicsolutions.com
see license attached to distribution
**************************************************************************/
##########SVN Build Data##########
##                              ##
## This File's Revision:        ##
##  $Rev:: 21338              $ ##
## File last change date:       ##
##  $Date:: 2011-05-03 14:04:#$ ##
##                              ##
##################################

/**
 * Smarty could be included via the autoloader in app_top.common.php, but still
 * requiring it here anyways so it's slightly less magical and less confusion for
 * people trying to troubleshoot.
 * 
 * NOTE:  There is a version of Smarty 3.0 in the php5_classes folder, that will
 * be used for 6.0.0, if you change the below to use the 3.0 version it will break
 * things as 5.2.* is not compatible with Smarty 3.0.
 */
require_once CLASSES_DIR.'smarty/Smarty.class.php';

/**
 * Template object that extends the Smarty class (a 3rd party library) to enable
 * using templates to display things.
 * 
 * @package System
 * @since Version 4.0.0
 */
class geoTemplate extends Smarty {
	/**
	 * G-Type:  Addon
	 */
	const ADDON = 'addon';
	
	/**
	 * G-Type:  Admin
	 */
	const ADMIN = 'admin';
	
	/**
	 * G-Type:  Module
	 */
	const MODULE = 'module';
	
	/**
	 * G-Type:  System
	 */
	const SYSTEM = 'system';
	
	/**
	 * G-Type:  External
	 */
	const EXTERNAL = 'external';
	
	/**
	 * G-Type:  Main_page
	 */
	const MAIN_PAGE = 'main_page';
	
	
	/**
	 * What this is set to determines the search behavior and where the
	 * template file is located.
	 *
	 * @var string One of: self::ADMIN, self::MODULE, self::SYSTEM, self::ADDON,
	 *  self::MAIN_PAGE, or self::EXTERNAL
	 */
	protected $_g_template_type;
	
	/**
	 * This should be "where" the template is used from.  For example, if
	 * using a template in the file classes/Cart.class.php the resource might be "cart".
	 *
	 * @var string
	 */
	protected $_g_resource;
	
	/**
	 * The page currently being loaded.  Mostly used when the template type is "main_page"
	 *
	 * @var int
	 */
	protected static $_g_page;
	
	/**
	 * The current language ID.
	 *
	 * @var int
	 */
	protected static $_g_language;
	
	/**
	 * The current category id.
	 *
	 * @var int
	 */
	protected static $_g_category;
	
	/**
	 * An array of "template sets" that will be searched through when looking
	 * for a template.  The array is sorted by key prior to doing template searching,
	 * so a key of "default" will always be last when the rest of the keys are
	 * numeric.
	 *
	 * @var array
	 */
	protected static $_g_template_sets = array('default' => 'default');
	
	/**
	 * Array of reserved names, any template sets using these names will be ignored
	 * @var array
	 */
	private static $_invalidTSetNames = array(
		'all',
		'active',
		'merged',
		'_temp'
	);
	
	/**
	 * Adds a template set to the template sets that it will retrieve templates from.
	 *
	 * @param string $name The same as the folder name, only alpha-numeric plus -_ and . (so no spaces)
	 */
	public static function addTemplateSet($name)
	{
		$name = self::cleanTemplateSetName($name);
		if (!$name) {
			//not a valid tempalte set name.
			return;
		}
		//add it to the array numerically.  Later the array will be sorted, so since the
		//default has index of "default", it will be pushed to the end of the array.
		if (!in_array($name,self::$_g_template_sets)) {
			self::$_g_template_sets[] = $name;
		}
	}
	/**
	 * Cleans the template set name given and makes sure it is "valid", if not
	 * valid it returns an empty string.  If it starts with _ or . or is in
	 * the list of invalid names, the name is considered invalid.
	 * 
	 * @param string $name
	 * @return string Returns empty string if string given is "ivalid".
	 */
	public static function cleanTemplateSetName ($name)
	{
		$name = preg_replace('/[^-a-zA-Z0-9_\.]+/','',$name);
		if (in_array($name, self::$_invalidTSetNames) || in_array(substr($entry,0,1), array ('_','.'))) {
			//invalid name, either it is reserved, or it starts with . or _
			return '';
		}
		return $name;
	}
	
	/**
	 * Gets the array of template sets in the order they are used
	 *
	 * @return array
	 */
	public static function getTemplateSets()
	{
		self::loadTemplateSets();
		return self::$_g_template_sets;
	}
	
	/**
	 * Gets a list of template set names that are "reserved names" which will be
	 * ignored by the template system.
	 * @return array
	 * @since Version 5.0.0
	 */
	public static function getInvalidSetNames ()
	{
		//TODO: Add some way for addons to add invalid tempalte set names perhaps?
		return self::$_invalidTSetNames;
	}
	
	/**
	 * Parses the text for any {external file='url'} type tags, and replaces
	 * them with the URL, then returns the parsed text.  This is good for
	 * allowing the use of {external ...} tags inside of text.
	 * 
	 * @param string $text
	 * @return string
	 * @since Version 5.0.0
	 */
	public static function parseExternalTags ($text)
	{
		if (strpos($text, '{external') === false) {
			//nothing found
			return $text;
		}
		//replace {external file="TEXT"} with call to geoTemplate::getUrl('', 'TEXT');
		$text = preg_replace('/\{external [^\}]*?file=(\'|")([^\'"]+)(\'|")[^}]*\}/ie', 'geoTemplate::getUrl(\'\', stripslashes(\'$2\'))', $text);
		return $text;
	}
	
	/**
	 * Specifies that the current template is an admin template.  The admin templates are a much
	 * simplified version of the front side, in that there is no "resource" to specify, and there
	 * is no template sets, all templates are in the admin/templates/ directory, and when
	 * calling fetch() the filename should be relative to that dir.
	 *
	 * @return geoTemplate Returns an instance of itself to allow chaining
	 */
	public function setAdmin()
	{
		$this->_g_template_type = self::ADMIN;
		//allow chaining
		return $this;
	}
	
	/**
	 * This template is for the specified module, and will be located under the module/module_name/ directory.
	 *
	 * @param string $module_filename The filename of the module (usually without the module_ before it)
	 * @return geoTemplate|bool Returns itself if successful, false otherwise.
	 */
	public function setModule ($module_filename)
	{
		//take off the end .php
		$name = str_replace('.php','',$module_filename);
		//make sure nothing invalid...
		$name = preg_replace('/[^a-zA-Z0-9-_\.]+/','',$name);
		
		if (strlen($name) == 0) {
			//invalid name
			return false;
		}
		$this->_g_template_type = self::MODULE;
		$this->_g_resource = $name;
		//allow chaining
		return $this;
	}
	
	/**
	 * This is a system template, used somewhere like browse_ads or the cart or somewhere.
	 *
	 * @param string $system_filename The system file's filename, without the .class and all lowercase.  If
	 *  the file resides in a sub-directory of the classes folder (for instance, one of the payment gateways
	 *  or order items), use the folder name instead (example: "order_item")
	 * @return boolean True if successful, false otherwise.
	 */
	public function setSystem ($system_filename)
	{
		//take off the end .php
		$name = str_replace('.php','',$system_filename);
		//make sure nothing invalid...
		$name = preg_replace('/[^a-zA-Z0-9-_\.]+/','',$name);
		
		if (strlen($name) == 0) {
			//invalid name
			return false;
		}
		$this->_g_template_type = self::SYSTEM;
		$this->_g_resource = $name;
		//allow chaining
		return $this;
	}
	
	/**
	 * This template is for the specified addon, and is located in one of 2 places:
	 * - geo_templates/template_set_name/addon/addon_name/ directory (searched first, to allow designers to include templates that override an addon's defaults)
	 * - addons/addon_name/templates/ directory
	 * 
	 * Note that for optimization reasons, it does not verify that the specified addon exists and is enabled, since
	 * typically only the addon itself will be using templates for the addon.
	 *
	 * @param string $addon_name The addon name, same as the addon's folder name and as set in the addon's info class.
	 * @return geoTemplate|bool Returns itself if successful (to allow chaining), false if addon name is not valid
	 */
	public function setAddon($addon_name)
	{
		//Don't bother checking if an addon is enabled, just make sure the name is valid
		//to prevent any funny business, like escaping the directory tree
		$addon_name = preg_replace('/[^a-zA-Z0-9-_]+/','',$addon_name);
		
		if (strlen($addon_name) == 0) {
			//name no good
			return false;
		}
		
		//if setting the addon, that means the type is addon
		$this->_g_template_type = self::ADDON;
		
		$this->_g_resource = $addon_name;
		return $this;
	}
	
	/**
	 * Set the page ID used for displaying a template in the main_page.
	 * 
	 * @param int|string $page_id
	 * @return geoTemplate Returns itself for easy chaining.
	 */
	public function setMainPage ($page_id)
	{
		self::setPage($page_id);
		$this->_g_template_type = self::MAIN_PAGE;
		//allow chaining
		return $this;
	}
	
	/**
	 * Not used yet.  Or is it?
	 *
	 * @param int|string $page_id
	 */
	public static function setPage($page_id)
	{
		$page_id = trim($page_id);
		
		self::$_g_page = $page_id;
	}
	
	/**
	 * Used to set the language ID (going to a Geo language ID) for templates.
	 *
	 * @param int $lang_id
	 */
	public static function setLanguage($lang_id)
	{
		$lang_id = intval($lang_id);
		if (!$lang_id) {
			$lang_id = 1; //default to 1
		}
		self::$_g_language = $lang_id;
	}
	
	/**
	 * Used to set category ID if displaying something specific for a category.
	 *
	 * @param int $category_id
	 */
	public static function setCategory ($category_id)
	{
		$category_id = intval($category_id);
		self::$_g_category = $category_id;
	}
	
	/**
	 * Creates a new template object, and assigns values for things that the majority of templates can use,
	 * like text messages assigned to smarty variable messages.  It also registers the following modifiers:
	 * fromDb: filters text that is coming from the database
	 * displayPrice: converts a number be formatted according to pre and post currency settings
	 * format_date: takes unix timestamp, and the specified format, and does date on it.
	 *
	 * @param string $template_type If specified, one of: ("module","system","addon","admin")
	 * @param string $resource Only used if template_type is specified and valid, and is not "admin".  
	 *  It is The resource that this template is used from, for example if using this template inside
	 *  classes/browse_ads.php the resource would be "browse_ads".
	 */
	public function __construct($template_type = null, $resource = null)
	{
		if ($template_type===null && defined('IN_ADMIN')) {
			$template_type = self::ADMIN;
		}
		
		//load all the template sets to choose from
		self::loadTemplateSets();
		if (isset($template_type)) {
			switch ($template_type) {
				case self::MODULE:
					$this->setModule($resource);
					break;
					
				case self::SYSTEM:
					$this->setSystem($resource);
					break;
					
				case self::ADDON:
					$this->setAddon($resource);
					break;
					
				case self::ADMIN:
					$this->setAdmin();
					break;
					
				case self::MAIN_PAGE:
					$this->setMainPage($resource);
					
				default:
					break;
			}
		}
		//set compile dir, be sure to take off the ending slash
		$this->compile_dir = rtrim(GEO_TEMPLATE_COMPILE_DIR,'/');
		
		/**
		 * Assign common variables that will be accessible from
		 * all templates.
		 */
		$db = DataAccess::getInstance();
		$session = geoSession::getInstance();
		
		$this->assign('messages',$db->get_text(true));
		
		//let templates know if logged in or not
		$logged_in = ($session->getUserId() > 0)? 1: 0;
		$this->assign('logged_in',$logged_in);
		
		//also let them know about common user info
		$user = array();
		$user['id'] = $session->getUserId();
		if ($logged_in) {
			//if logged in, also do the username
			$user['username'] = $session->getUserName();
		}
		$user['detected_robot'] = $session->is_robot();
		$this->assign('user',$user);
		
		//let template know the current language too
		if (isset(self::$_g_language)) {
			$this->assign('language_id',self::$_g_language);
		} else {
			//grab the language from the session object instead
			$this->assign('language_id',$session->getLanguage());
		}
		
		if (isset(self::$_g_page)) {
			//current page if already set
			$this->assign('page_id',self::$_g_page);
		}
		
		if (isset(self::$_g_category)) {
			//current category if already set
			$this->assign('category_id',self::$_g_category);
			$catId = self::$_g_category;
			//get all the parents
			$parents = array ();
			$sql = "SELECT `parent_id` FROM ".geoTables::categories_table." WHERE `category_id`=?";
			
			while ($catId) {
				$row = $db->GetRow($sql, array($catId));
				if ($row && $row['parent_id']) {
					if (isset($parents[$row['parent_id']])) {
						//stop potential infinite loop on mis-configured sites
						break;
					}
					$parents[$row['parent_id']] = $row['parent_id'];
				}
				$catId = (int)$row['parent_id'];
			}
			$this->assign('parent_categories', $parents);
		}
		$this->assign('classifieds_url',$db->get_site_setting('classifieds_url'));
		$this->assign('site_base_url',self::getBaseUrl());
		$this->assign('classifieds_file_name',$db->get_site_setting('classifieds_file_name'));
		if (geoPC::is_ent()) {
			$this->assign('affiliate_url', $db->get_site_setting('affiliate_url'));
		}
		
		/**
		 * Register modifiers, filters, and functions to be used in
		 * the smarty templates...
		 */
		
		//modifier: fromDB - runs string through geoString::fromDB()
		$this->register_modifier('fromDB',array('geoString','fromDB'));
		
		//modifier: displayPrice - run a number through geoString::displayPrice()
		$this->register_modifier('displayPrice', array('geoString','displayPrice'));
		
		//modifier: format_date - translate date
		$this->register_modifier('format_date',array('geoTemplate','formatDate'));
		
		//modifier: escape_js - run string through filters so it's safe to put into JS surrounded by single quotes
		$this->register_modifier('escape_js', array('geoTemplate', 'escapeJs'));
		
		//function: body_html - makes {body_html} work
		$this->register_function('body_html',array('geoTemplate','getBodyHtml'));
		
		//function: header_html - makes {header_html} work
		$this->register_function('header_html',array('geoTemplate','getHeaderHtml'));
		
		//function: module - makes {module ...} work
		$this->register_function('module',array('geoTemplate','getModuleHtml'));
		
		//function: external - makes {external ...} work
		$this->register_function('external',array('geoTemplate','getExternalUrl'));
		
		//function: addon - makes {addon ...} work
		$this->register_function('addon',array('geoTemplate','getAddonHtml'));
		
		//Filter: if in demo mode, filter out all e-mail addresses
		if (defined('DEMO_MODE')) {
			$view = geoView::getInstance();
			if (!$view->allowEmail && defined('IN_ADMIN')) {
				$this->register_outputfilter(array('geoTemplate','stripEmails'));
			}
		}
		if (geoPC::is_trial()) {
			if (!defined('IN_ADMIN')) {
				$this->register_outputfilter(array('geoTemplate','addPoweredBy'));
				
				$t_sets = self::getTemplateSets();
				if (count($t_sets) > 1) {
					//turn on security, prevent arbitrary code execution
					$this->security = true;
					
					//Add security locations
					$secure_dirs = array();
					if (defined('IN_ADMIN')) {
						$secure_dirs[] = ADMIN_DIR.'templates/';
					}
					$secure_dirs[] = GEO_TEMPLATE_DIR;
					
					//add addon template locations
					$addon = geoAddon::getInstance();
					$list = $addon->getEnabledList();
					foreach ($list as $name => $data) {
						if (file_exists(ADDON_DIR . $name . '/templates/')) {
							$secure_dirs[] = ADDON_DIR . $name . '/templates/';
						}
					}
					
					$this->secure_dir = $secure_dirs;
				}
			}
			$this->assign('isTrial',1);
		} else {
			$this->assign('isTrial',0);
		}
		//filters: Make addon filter_display_page and filter_display_page_nocache work
		if (!defined('IN_ADMIN') && !defined('IN_GEO_RSS_FEED')) {
			$addon = geoAddon::getInstance();
			if ($addon->coreEventCount('filter_display_page') > 0) {
				$this->register_outputfilter(array('geoTemplate','filterPage'));
			}
			if ($addon->coreEventCount('filter_display_page_nocache') > 0) {
				$this->register_outputfilter(array('geoTemplate','filterPageNocache'));
			}
			//for internal use: demo box
			if (defined('DEVELOPER_MODE')) {
				$this->register_outputfilter(array('geoTemplate','demoBox'));
			}
		}
	}
	
	/**
	 * Convenience method, to get the base URL based on settings set in the
	 * admin.  If currently using SSL connection, it uses SSL URL from admin
	 * settings.
	 * 
	 * @return string
	 */
	public static function getBaseUrl ()
	{
		$urlSetting = (geoSession::isSSL())? 'classifieds_ssl_url' : 'classifieds_url';
		return dirname(DataAccess::getInstance()->get_site_setting($urlSetting)).'/';
	}
	
	protected static $_g_loadTemplateSetsRun = false;
	
	/**
	 * Loads the template sets to use by including the file geo_templates/t_sets.php
	 * 
	 * That file should be generated by the admin template control panel
	 *
	 * @param bool $force_reload If true, will force it to re-load the template sets even
	 *  if it was already run in this page load.
	 */
	public static function loadTemplateSets($force_reload = false)
	{
		if (!self::$_g_loadTemplateSetsRun || $force_reload) {
			if ($force_reload) {
				//reset the sets, usually this only happens when the sets are
				//being changed, like in the admin.
				self::$_g_template_sets = array ('default' => 'default');
			}
			if (file_exists(GEO_TEMPLATE_DIR . 't_sets.php')) {
				include GEO_TEMPLATE_DIR . 't_sets.php';
			}
			
			self::$_g_loadTemplateSetsRun = true;
			//allow addons to add template sets if they wish
			geoAddon::triggerUpdate('notify_geoTemplate_loadTemplateSets', array('force_reload'=>$force_reload));
		}
		//be sure to sort the array by key so that default always ends up last
		ksort(self::$_g_template_sets,SORT_STRING);
	}
	
	private static $_inBodyHtml = false;
	
	/**
	 * Smarty Function: Used to replace {body_html} which loads the dynamic part
	 * of each page
	 *
	 * @param array $params
	 * @param geoTemplate $smarty
	 * @return string
	 */
	public static function getBodyHtml ($params, $smarty)
	{
		$return = '';
		if (self::$_inBodyHtml) {
			//prevent infinite recursion, we are already inside of a body_html
			return '';
		}
		self::$_inBodyHtml = true;
		if (isset($smarty->_tpl_vars['geo_inc_files']['body_html'])) {
			//include template
			$vars = $params;
			$params['smarty_include_tpl_file'] = $smarty->_tpl_vars['geo_inc_files']['body_html'];
			
			if (isset($smarty->_tpl_vars['body_vars'])) {
				//vars set in TPL file over-write vars as set in PHP file...
				$vars = array_merge($smarty->_tpl_vars['body_vars'], $vars);
			}
			if (isset($smarty->_tpl_vars['geo_inc_files']['body_html_addon'])) {
				$vars['g_type'] = (isset($vars['g_type']))? $vars['g_type'] : self::ADDON;
				$vars['g_resource'] = (isset($vars['g_resource']))? $vars['g_resource'] : $smarty->_tpl_vars['geo_inc_files']['body_html_addon'];
			} else if (isset($smarty->_tpl_vars['geo_inc_files']['body_html_system'])) {
				$vars['g_type'] = (isset($vars['g_type']))? $vars['g_type'] : self::SYSTEM;
				$vars['g_resource'] = (isset($vars['g_resource']))? $vars['g_resource'] : $smarty->_tpl_vars['geo_inc_files']['body_html_system'];
			}
			$params['smarty_include_vars'] = $vars;
			$smarty_template_vars = $smarty->_tpl_vars;
			$smarty->_smarty_include($params);
			$smarty->_tpl_vars = $smarty_template_vars;
		} 
		if (isset($smarty->_tpl_vars['body_html'])) {
			//In addition to (not instead of), if body html is set, append that too
			$return .= $smarty->_tpl_vars['body_html'];
		}
		//allows multiple body_html on page, just not inside each other...
		self::$_inBodyHtml = false;
		return $return;
	}
	
	private static $_inHeaderHtml = false;
	
	/**
	 * Used by "custom" smart tag {header_html} to display the dynamic part of
	 * stuff that should be in the <head></head> section of the page.
	 * 
	 * @param array $params
	 * @param geoTemplate $smarty
	 * @return string
	 */
	public static function getHeaderHtml ($params, $smarty)
	{
		if (self::$_inHeaderHtml) {
			//prevent infinite recursion, we are already inside of a header_html
			return '';
		}
		self::$_inHeaderHtml = true;
		if (isset($smarty->_tpl_vars['_header_html'])) {
			//In addition to (not instead of), if header html is set, append
			//that too.  Unlike body HTML, the header html goes before the
			//header template, that way JS and CSS files can be loaded first.
			echo $smarty->_tpl_vars['_header_html'];
		}
		if (isset($smarty->_tpl_vars['geo_inc_files']['header_html'])) {
			//include template
			$vars = $params;
			$params['smarty_include_tpl_file'] = $smarty->_tpl_vars['geo_inc_files']['header_html'];
			
			if (isset($smarty->_tpl_vars['header_vars'])) {
				//vars set in TPL file over-write vars as set in PHP file...
				$vars = array_merge($smarty->_tpl_vars['header_vars'], $vars);
			}
			if (isset($smarty->_tpl_vars['geo_inc_files']['header_html_addon'])) {
				$vars['g_type'] = (isset($vars['g_type']))? $vars['g_type'] : self::ADDON;
				$vars['g_resource'] = (isset($vars['g_resource']))? $vars['g_resource'] : $smarty->_tpl_vars['geo_inc_files']['header_html_addon'];
			} else if (isset($smarty->_tpl_vars['geo_inc_files']['header_html_system'])) {
				$vars['g_type'] = (isset($vars['g_type']))? $vars['g_type'] : self::SYSTEM;
				$vars['g_resource'] = (isset($vars['g_resource']))? $vars['g_resource'] : $smarty->_tpl_vars['geo_inc_files']['header_html_system'];
			}
			$params['smarty_include_vars'] = $vars;
			
			$smarty_template_vars = $smarty->_tpl_vars;
			$smarty->_smarty_include($params);
			$smarty->_tpl_vars = $smarty_template_vars;
		} 
		self::$_inHeaderHtml = false;
		return '';
	}
	
	public static function getExternalUrl ($params, $smarty)
	{
		$file = geoFile::cleanPath($params['file']);
		$g_resource = isset($params['g_resource'])? $params['g_resource']: '';
		
		return self::getUrl($g_resource, $file);
	}
	
	/**
	 * Used by "custom" smart tag {module ...} to display the contents of
	 * a particular module.
	 * 
	 * @param array $params
	 * @param geoTemplate $smarty
	 * @return string
	 */
	public static function getModuleHtml ($params, $smarty)
	{
		$return = '';
		//check to make sure all the parts are there
		if (!isset($params['tag'])) {
			//tag not specified
			return '{module tag syntax error}';
		}
		$tag = $params['tag'];
		
		if (isset($smarty->_tpl_vars['geo_inc_files']['modules'][$tag])) {
			//include template, passing along any vars set in module tag
			$vars = $params;
			
			//cache output
			$pageCache = geoCachePage::getInstance();
			
			$params['smarty_include_tpl_file'] = $smarty->_tpl_vars['geo_inc_files']['modules'][$tag];
			if (isset($smarty->_tpl_vars['module_vars'][$tag])) {
				//vars set in TPL file over-write vars as set in PHP file...
				$vars = array_merge($smarty->_tpl_vars['module_vars'][$tag], $vars);
			}
			$vars['g_type'] = (isset($vars['g_type']))? $vars['g_type'] : self::MODULE;
			$vars['g_resource'] = (isset($vars['g_resource']))? $vars['g_resource'] : $tag;
			
			$params['smarty_include_vars'] = $vars;
			$smarty_template_vars = $smarty->_tpl_vars;
			
			if ($pageCache->canCache($tag)) {
				//do our own brand of caching
				ob_start();
			}
			$smarty->_smarty_include($params);
			if ($pageCache->canCache($tag)) {
				//do our own brand of caching, don't use smarty's as that would
				//require yet another folder to be writable
				$cacheTxt = ob_get_contents();
				ob_end_clean();
				$language_id = $smarty->_tpl_vars['language_id'];
				$cat_id = (isset($smarty->_tpl_vars['category_id']))? $smarty->_tpl_vars['category_id']: 0;
				$logged_in = $smarty->_tpl_vars['logged_in'];
				$pageCache->update($tag, $language_id, $cat_id, $logged_in, geoCachePage::quotePage($cacheTxt));
				echo $cacheTxt;
			}
			$smarty->_tpl_vars = $smarty_template_vars;
		} 
		if (isset($smarty->_tpl_vars['modules'][$tag]['body'])) {
			//In addition to (not instead of), if text is set, append that too
			$return .= $smarty->_tpl_vars['modules'][$tag]['body'];
		}
		return $return;
	}
	
	/**
	 * Used by "custom" smart tag {addon ...} to display something for an addon
	 * tag.
	 * 
	 * @param array $params
	 * @param geoTemplate $smarty
	 * @return string
	 */
	public static function getAddonHtml ($params, $smarty)
	{
		$return = '';
		//check to make sure all the parts are there
		if (!isset($params['author'], $params['addon'], $params['tag'])) {
			//tag not specified
			return '{addon tag syntax error}';
		}
		$auth = $params['author'];
		$addon = $params['addon'];
		$tag = $params['tag'];
		
		if (isset($smarty->_tpl_vars['geo_inc_files']['addons'][$auth][$addon][$tag])) {
			//include template
			$vars = $params;
			$params['smarty_include_tpl_file'] = $smarty->_tpl_vars['geo_inc_files']['addons'][$auth][$addon][$tag];
			
			if (isset($smarty->_tpl_vars['addon_vars'][$auth][$addon][$tag])) {
				//vars set in TPL file over-write vars as set in PHP file...
				$vars = array_merge($smarty->_tpl_vars['addon_vars'][$auth][$addon][$tag], $vars);
			}
			$vars['g_type'] = (isset($vars['g_type']))? $vars['g_type'] : self::ADDON;
			$vars['g_resource'] = (isset($vars['g_resource']))? $vars['g_resource'] : $addon;
			
			$params['smarty_include_vars'] = $vars;
			
			$smarty_template_vars = $smarty->_tpl_vars;
			$smarty->_smarty_include($params);
			$smarty->_tpl_vars = $smarty_template_vars;
		} 
		if (isset($smarty->_tpl_vars['addons'][$auth][$addon][$tag]['body'])) {
			//In addition to (not instead of), if module text is set, append that too
			$return .= $smarty->_tpl_vars['addons'][$auth][$addon][$tag]['body'];
		}
		return $return;
	}
	
	/**
	 * Smarty modifier: Alternative to smarties' date_format modifier, this one 
	 * uses the date() function instead of strftotime like the smarty equivelant does.
	 *
	 * @param int $date
	 * @param string $format
	 * @return string
	 */
	public static function formatDate($date, $format=null)
	{
		if ($format === null) {
			$format = DataAccess::getInstance()->get_site_setting('entry_date_configuration');
		}
		return date($format,$date);
	}
	
	//escapeJs
	/**
	 * Smarty modifier: Filters text to be suitable for use in JS
	 *
	 * @param string $string
	 * @return string
	 */
	public static function escapeJs($string)
	{
		//equivelent of using smarty modifiers: |regex_replace:"/[\r\n\t]/":' '|addslashes}
		$string = preg_replace('/[\r\n\t]+/', ' ', $string);
		$string = addslashes($string);
		return $string;
	}
	
	/**
	 * Smarty pre-filter: used to pre-filter templates through addon core event
	 * filter_display_page
	 *
	 * @param string $tpl_source
	 * @param geoTemplate $smarty
	 * @return string
	 */
	public static function filterPage($tpl_source, $smarty)
	{
		//smarty pre-filter
		return geoAddon::triggerDisplay('filter_display_page',$tpl_source, geoAddon::FILTER);
	}
	
	/**
	 * Smarty output filter: used to filter the output of any templates, by
	 * running it through the addon core event filter_display_page_nocache
	 *
	 * @param string $tpl_source
	 * @param geoTemplate $smarty
	 * @return string
	 */
	public static function filterPageNocache($tpl_source, $smarty)
	{
		//smarty output filter
		return geoAddon::triggerDisplay('filter_display_page_nocache',$tpl_source, geoAddon::FILTER);
	}
	
	/**
	 * over-writes smarties' built-in functionality, to determine the absolute server path
	 * for the template location using getFilePath() and passes it on to smarty parent
	 * function.
	 *
	 * @param array $params
	 */
	public function _smarty_include($params)
	{
		//echo "params:<pre>".print_r($params,1)."</pre><br />";
		
		$old_g_type = $this->_g_template_type;
		$old_g_resource = $this->_g_resource; 
		
		$g_type = (isset($params['smarty_include_vars']['g_type']))? $params['smarty_include_vars']['g_type']: $this->_g_template_type;
		$g_resource = (isset($params['smarty_include_vars']['g_resource']))? $params['smarty_include_vars']['g_resource']: $this->_g_resource;
		$params['smarty_include_tpl_file'] = self::getFilePath($g_type,$g_resource,$params['smarty_include_tpl_file']);
		
		//Set g type and g resource of this object while we are in this file
		$this->_g_template_type = $g_type;
		$this->_g_resource = $g_resource;
		
		parent::_smarty_include($params);
		
		//restore original gtype and g resource
		$this->_g_template_type = $old_g_type;
		$this->_g_resource = $old_g_resource;
	}
	
	/**
	 * Run-time cache used so we don't have to scan for a certain template more than once in a page load.
	 *
	 * @var array
	 */
	protected static $_filePaths;
	
	/**
	 * Gets the absolute path to the template file based on the template type, the resource, and the template filename.
	 * It will go through each of the template sets until it finds the file, if it doesn't find it in one template set,
	 * it looks at the next template set in the list, ending with the default template set.  If the template type is
	 * addon, and it hasn't found the template in any of the template sets under template_set/addon/addon_name/template_filename
	 * it then looks in addons/addon_name/templates/template_name 
	 *
	 * @param string $g_type The type of template, either system, addon, admin, or module
	 * @param string $g_resource Typically the file name that the template is used in
	 * @param string $template_filename The filename, if none specified it assumes index.tpl.  Note that the
	 *  template_filename can include directories, but must be relative to the resources directory.  If set to
	 *  index, and it finds the file with the resource name.tpl it will use that.
	 * @param bool $dieOnError If false, will NOT die if the file path is not found, instead
	 *   it would continue and just return the path location that is attempting to be found.
	 * @param bool $falseOnError If true and dieOnError is false, will return false if file not found.
	 * @return string The absolute path to the template file, or the original tempalte_filename specified
	 *  if the absolute path could not be found.
	 */
	public static function getFilePath($g_type, $g_resource, $template_filename = 'index', $dieOnError = true, $falseOnError = false)
	{
		self::loadTemplateSets();
		if (!isset(self::$_filePaths)) {
			//attempt to load cached file paths from geo cache system.
			$file_cache = geoCacheSetting::getInstance()->process('smarty_template_paths');
			if (!is_array($file_cache)) {
				$file_cache = array();
			}
			self::$_filePaths = $file_cache;
		}
		
		if ($g_type != self::EXTERNAL && strpos($template_filename,'.tpl') === false && preg_match ('/\.[a-zA-Z]{3,4}$/',$template_filename) == 0) {
			//add the .tpl to the end if it's not there, and it does not use another 3 or 4 letter extension
			//such as .php, or .html
			$template_filename .= '.tpl';
		}
		if (isset(self::$_filePaths[$g_type][$g_resource][$template_filename])) {
			//path already known!  We don't have to do a bunch of scanning for files now...
			//Note that it does NOT call file_exists() on purpose, in order to speed
			//things up.  This may result in situation where the cache is out of date
			//and has old locations for files, in that case the cache needs to be reset in
			//the admin.
			return self::$_filePaths[$g_type][$g_resource][$template_filename];
		}
		
		$path_to_use = false;
		if ($g_type == self::ADMIN) {
			//all admin templates are in the admin/templates/ directory.  There is no seperate template set.
			$path_to_use = ADMIN_DIR . "templates/$template_filename";
		} else {
			$t_set = self::whichTemplateSet($g_type, $g_resource, $template_filename);
			
			if ($t_set && ($g_type !== self::ADDON || $t_set !== 'default')) {
				//if the t_set was found, and the type was not addon, or if
				//it was addon, the tset is not default (don't pull any files
				//from default/addon/ to make things easier on addon developers)
				$g_resource_dir = (strlen($g_resource) > 0)? "$g_resource/": '';
				$resource = GEO_TEMPLATE_DIR . "$t_set/$g_type/{$g_resource_dir}$template_filename";
				if (!file_exists($resource)) {
					//if it doesn't exist, it must be one of them fancy ones
					$resource = GEO_TEMPLATE_DIR . "$t_set/$g_type/{$g_resource}.tpl";
				}
				$path_to_use = $resource;
			} else if ($g_type === self::ADDON) {
				//This is an addon, and the template wasn't found in any of the
				//template sets, so see if the file exists in:
				//addons/addon_name/templates/template_filename.tpl
				$resource = ADDON_DIR . $g_resource . "/templates/$template_filename";
				if (file_exists($resource)) {
					$path_to_use = $resource;
				}
			}
		}
		if (!$path_to_use) {
			//path couldn't be found, return the original template filename passed in
			$msg = "ERROR TEMPLATE: Template could not be found for: g_type: $g_type g_resource: $g_resource template_filename: $template_filename";
			//die($msg); //un-comment to easily find when templates dont work.
			trigger_error($msg);
			
			if ($dieOnError) {
				//go ahead and display an error on the page, instead of just causing
				//a fatal error.
				if ($g_resource) $g_type .= '/'.$g_resource;
				self::template404("$g_type/$template_filename");
			}
			if ($falseOnError) {
				return false;
			}
			$path_to_use = $template_filename;
		}
		
		//remember it for the rest of the page load, so if the same template is used
		//again we don't have to re-scan everything
		self::$_filePaths[$g_type][$g_resource][$template_filename] = $path_to_use;
		
		if (geoCache::get('use_cache')) {
			//If cache is turned on, cache the location too so we don't have to scan after this.
			geoCacheSetting::getInstance()->update('smarty_template_paths',self::$_filePaths);
		}
		
		return $path_to_use;
	}
	
	/**
	 * Run-time cache used so we don't have to scan for a certain template more than once in a page load.
	 *
	 * @var array
	 */
	protected static $_fileSets;
	
	/**
	 * Gets which template set the the template file is in based on the template type, the resource, and the template filename.
	 * It will go through each of the template sets until it finds the file, if it doesn't find it in one template set,
	 * it looks at the next template set in the list, ending with the default template set.
	 * 
	 * If the file can not be found in any template sets, returns empty string.
	 *
	 * @param string $g_type The type of template, either self::SYSTEM, self::ADDON, self::MODULE, 
	 *  self::MAIN_PAGE, or self::EXTERNAL
	 * @param string $g_resource Typically the file name that the template is used in
	 * @param string $template_filename The filename, if none specified it assumes index.tpl.  Note that the
	 *  template_filename can include directories, but must be relative to the resource's directory.  If set to
	 *  index, and it finds the file with the resource name.tpl it will use that.
	 * @return string The name for the template set where the template was found, or empty string if not
	 *  found.
	 */
	public static function whichTemplateSet ($g_type, $g_resource, $template_filename = 'index')
	{
		self::loadTemplateSets();
		if (!isset(self::$_fileSets)) {
			//attempt to load cached file paths from geo cache system.
			$file_cache = geoCacheSetting::getInstance()->process('smarty_template_file_sets');
			if (!is_array($file_cache) || defined('THEME_PRIMARY') || defined('THEME_SECONDARY')) {
				//either it isn't an array, or we are using demo theme selector so don't want to use cache for this
				$file_cache = array();
			}
			self::$_fileSets = $file_cache;
		}
		
		if ($g_type != self::EXTERNAL && strpos($template_filename,'.tpl') === false && preg_match ('/\.[a-zA-Z]{3,4}$/',$template_filename) == 0) {
			//add the .tpl to the end if it's not there
			$template_filename .= '.tpl';
		}
		if (isset(self::$_fileSets[$g_type][$g_resource][$template_filename])) {
			//path already known!  We don't have to do a bunch of scanning for files now...
			//Note that it does NOT call file_exists() on purpose, in order to speed
			//things up.  This may result in situation where the cache is out of date
			//and has old locations for files, in that case the cache needs to be reset in
			//the admin.
			return self::$_fileSets[$g_type][$g_resource][$template_filename];
		}
		
		ksort(self::$_g_template_sets,SORT_STRING);
		$template_sets = self::$_g_template_sets;
		$res_ends = array();
		//Typically, the template will be name something like this:
		//type/resource/template_name.tpl
		$g_resource_dir = (strlen($g_resource) > 0)? "$g_resource/": '';
		$res_ends [] = "/{$g_type}/{$g_resource_dir}$template_filename";
		if ($template_filename == 'index.tpl') {
			//To cut down on un-necessary directories that only have 1 file in them,
			//if the resource is index - also allow it to be accessed using path similar to:
			//type/resource.tpl
			//instead of
			//type/resource/index.tpl
			$res_ends[] = "/{$g_type}/{$g_resource}.tpl";
		}
		$set_to_use = '';
		foreach ($template_sets as $template_set) {
			foreach ($res_ends as $res_end) {
				//This is rather expensive, especially the more "cache set" levels
				//the admin has specified.  We offset this by once we find the location
				//of a template, we remember where it is and save that using geoCache
				//if the geoCache is turned on.
				$resource = GEO_TEMPLATE_DIR . $template_set . $res_end;
				if (file_exists($resource)) {
					//this is the template set
					$set_to_use = $template_set;
					break;
				}
			}
			if ($set_to_use) {
				//location was found already
				break;
			}
		}
		
		//remember it for the rest of the page load, so if the same template is used
		//again we don't have to re-scan everything
		self::$_fileSets[$g_type][$g_resource][$template_filename] = $set_to_use;
		
		if (geoCache::get('use_cache') && !defined('THEME_PRIMARY') && !defined('THEME_SECONDARY')) {
			//If cache is turned on, cache the location too so we don't have to scan after this.
			//(Don't cache if using demo's theme selector though)
			geoCacheSetting::getInstance()->update('smarty_template_file_sets',self::$_fileSets);
		}
		
		return $set_to_use;
	}
	
	/**
	 * Gets the URL for "external" template files, meaning things like css or js files that are located
	 * in the template set.
	 * 
	 * @param string $g_resource Typically the file name that the template is used in
	 * @param string $template The filename, if none specified it returns the url to the
	 *  base of the templates directory, where all the template sets are located
	 * @return string The URL for where the file exists inside TEMPLATE_SET/external/resource/filename
	 *  or just resource/filename if file could not be found.
	 */
	public static function getUrl ($g_resource='', $filename = '', $emptyOnFailure = false)
	{
		if (strlen($filename) == 0) {
			return GEO_TEMPLATE_URL;
		}
		if (strpos($filename, 'http://') === 0 || strpos($filename, 'https://') === 0) {
			//it has http:// at front
			return $filename;
		}
		
		//search for what template set the file is in
		//Note: No need to cache in geoCache since whichTemplate already does
		$t_set = self::whichTemplateSet(self::EXTERNAL,$g_resource,$filename);
		$g_resource = ($g_resource)? $g_resource.'/': '';
		if (!$t_set) {
			//could not find template set, return relative
			return ($emptyOnFailure)? '' : $g_resource.$filename;
		}
		return GEO_TEMPLATE_URL . "$t_set/" . self::EXTERNAL . "/{$g_resource}$filename";
	}
	
	/**
	 * Strips the form tags, for use in the demo.
	 *
	 * @param string $tpl_source
	 * @param geoTemplate $smarty
	 * @return string
	 */
	public static function stripFormTags ($tpl_source, $smarty)
	{
		
		return preg_replace('/\<form[^>]*\>/ei','geoTemplate::_replaceFormTag("\\0")',$tpl_source);
	}
	
	private static function _replaceFormTag($form_tag)
	{
		if (strpos($form_tag,'id="switch_product"') === false) {
			//this is not the switch product form tag.
			$form_tag = '';
		}
		return $form_tag;
	}
	
	/**
	 * For internal use only, used to display the demo box at the top of the
	 * page on the demo.
	 * 
	 * @param $tpl_source
	 * @param $smarty
	 * @return string
	 * @access private
	 */
	public static function demoBox ($tpl_source, $smarty)
	{
		//For internal use: In demo mode, display demo box at top of page.
		if (defined('DEMO_THEME')) {
			$tpl_source = str_replace('geostyle.css',DEMO_THEME,$tpl_source);
		}
		return preg_replace('/(\<body[^>]*\>)/','$1'.DEVELOPER_MODE,$tpl_source);
	}
	
	/**
	 * For internal use only, used to strip all e-mails from being displayed
	 * on the demo, to prevent e-mail harvesting on our demos.
	 * 
	 * @param $tpl_source
	 * @param $smarty
	 * @return string
	 * @access private
	 */
	public static function stripEmails ($tpl_source, $smarty)
	{
		$replace_with_email = 'HIDDEN FOR DEMO';
		return preg_replace('/(^|[^a-zA-Z0-9]+)'.geoString::EMAIL_PREG_EXPR.'($|[^a-zA-Z])/','$1'.$replace_with_email.'$2',$tpl_source);
	}
	
	/**
	 * For internal use only, used to add "Powered By" text to trial installations.
	 * 
	 * @param $tpl_source
	 * @param $smarty
	 * @return string
	 * @access private
	 */
	public static function addPoweredBy ($tpl_source, $smarty)
	{
		$js = '<script type="text/javascript">if (top.location != location){top.location.href = document.location.href;}</script>';
		
		$trial = ' :: <span style="color: red;">TRIAL VERSION</span> :: ';
		
		$classOnly = 'http://geodesicsolutions.com/software/classifieds-only.html';
		$auctionOnly = 'http://geodesicsolutions.com/software/auctions-only.html';
		$home = 'http://geodesicsolutions.com/';
		
		$poweredBy = "$js<div style='text-align: center; background-color: white; color: black; font-weight: bold; margin: 10px; border: 1px solid black; padding: 5px;'>
		$trial<a href='$classOnly'>Classified Ad Software</a> and <a href='$auctionOnly'>Auction Software</a> Powered by <a href='$home'>Geodesic Solutions, LLC</a>$trial</div>";
		
		
		$search = array('/<\/body[^>]*\>/i','/<body([^>]*)\>/i');
		$replace = array ($poweredBy.'</body>', '<body\\1>'.$poweredBy);
		return preg_replace($search, $replace, $tpl_source);
	}
	
	/**
	 * Over-write the existing smarty functionality, so that it will determine the full path
	 * to the template file based on the template type and g_resource (different than resource_name
	 * as used by smarty)
	 *
	 * @param string $resource_name
	 * @param string $cache_id
	 * @param string $compile_id
	 * @param boolean $display
	 * @return mixed
	 */
	public function fetch($resource_name = 'index', $cache_id = null, $compile_id = null, $display = false)
	{
		if (!isset($this->_g_template_type)) {
			//must not be using new system, pass it directly on
			return parent::fetch($resource_name,$cache_id,$compile_id,$display);
		}
		$resource_name = self::getFilePath($this->_g_template_type,$this->_g_resource,$resource_name);
		return parent::fetch($resource_name,$cache_id,$compile_id,$display);
	}
	
	/**
	 * Displays an error and stops the rest of the page from loading.  Used when
	 * a template file cannot be found in any template sets.
	 * 
	 * @param string $tplFile The file that could not be found
	 * @param string $longMsg Optional additional info, if supplied it will be
	 *   displayed to the user.
	 * @since Version 5.0.0
	 */
	public static function template404 ($tplFile, $longMsg='')
	{
		//Die on error to prevent fatal errors from displaying.
		include GEO_BASE_DIR . 'app_bottom.php';
		
		if (strpos($tplFile,'.php') !== false) {
			$longMsg .= '<br /><br />The Admin may be able to fix this by using the <em>Re-Scan Attachments</em> tool for the template set.';
		}
		die ("<div style='border: 1px black solid; padding: 15px;'>
		<strong style='color:red;'>Template Error:</strong> Template file not found!<br /><br />
		The template file <strong>{$tplFile}</strong> could not be found in any
		of the template sets currently loaded.
		<br /><br />
		{$longMsg}
		</div>");
	}
	
	/**
	 * Allow object notation to be used to set smarty params.
	 *
	 * @param string $name
	 * @param mixed $value
	 */
	public function __set($name,$value){
		$this->assign($name, $value);
	}
}