<?php
//Listing.class.php
/**
 * Holds the geoListing object.
 * 
 * @package System
 * @since Version 4.0.0
 */
/**************************************************************************
Geodesic Classifieds & Auctions Platform 5.2
Copyright (c) 2001-2011 Geodesic Solutions, LLC
All rights reserved
http://geodesicsolutions.com
see license attached to distribution
**************************************************************************/
##########SVN Build Data##########
##                              ##
## This File's Revision:        ##
##  $Rev:: 20911              $ ##
## File last change date:       ##
##  $Date:: 2011-01-13 15:41:#$ ##
##                              ##
##################################

/**
 * A listing object, basically a container object for particular listing.
 * 
 * Since there are many listings, there can be many geoListing objects.  If you
 * want to get or set information for a listing, this is the object to use.
 * 
 * It uses the magic methods __get and __set which enable getting a field by
 * calling $listing->field or setting a field by using $listing->field = $value.
 * The fields are NOT automatically encoded or decoded using that, you will
 * need to encode or decode the values as necessary depending on the field.
 * 
 * @package System
 * @since Version 4.0.0
 */
class geoListing
{
	/**
	 * The listing data, un-encoded and everything.
	 * @var array
	 */
	private $_listing_data;
	
	/**
	 * Whether or not all the fields are retrieved, or just some of them.
	 * @var bool
	 */
	private $_all;
	
	/**
	 * Whether or not the current listing is in the expired table or not.
	 * @var bool
	 */
	private $_isExpired;
	
	/**
	 * Array of listing objects
	 * @var geoListing array
	 */
	private static $_listings;
	
	/**
	 * Convienience function, gets the title for the specified listing.  It
	 * decodes the title for you using fromDB.
	 * 
	 * If you know
	 * you will be needing more info that just the title, just get the whole
	 * listing instead, and decode it yourself.  This is kind of like calling
	 * geoString::fromDB(geoListing::getListing($listing_id)->title)
	 *
	 * @param int $listing_id
	 * @return string The title (already decoded from the db), or the listing ID if
	 *  the listing could not be found.
	 */
	public static function getTitle ($listing_id=0)
	{
		$listing = self::getListing($listing_id,false);
		if (is_object($listing)) {
			return geoString::fromDB($listing->title);
		}
		return $listing_id;
	}
	
	/**
	 * Gets the count of how many bids there currently are for the given
	 * listing ID.
	 * @param int $listing_id
	 * @return int|bool The current number of bids, or boolean false if there
	 *  was a problem getting the count.
	 */
	public static function getBids($listing_id)
	{
		$listing_id = (is_array($listing_id))? $listing_id: array($listing_id);
		$db = DataAccess::getInstance();
		$sql = "SELECT count(*) total_bids FROM geodesic_auctions_bids WHERE auction_id=?";
		$r = $db->GetRow($sql,$listing_id);
		if (!$r) {
			trigger_error('ERROR SQL: Query failed, sql: '.$sql.' Error: '.$db->ErrorMsg());
			return false;
		} else {
			return $r['total_bids'];
		}
	}
	
	/**
	 * Gets array of tags for the given listing ID
	 * @param int $listing_id
	 * @return array
	 * @since Version 5.1.0
	 */
	public static function getTags ($listing_id)
	{
		$listing_id = (int)$listing_id;
		if (!$listing_id) {
			return array();
		}
		
		$db = DataAccess::getInstance();
		
		$rows = $db->GetAll("SELECT `tag` FROM ".geoTables::tags." WHERE `listing_id`=$listing_id");
		$tags = array();
		foreach ($rows as $row) {
			$tags[] = geoString::fromDB($row['tag']);
		}
		return $tags;
	}
	
	/**
	 * Gets a listing according to the listing id specified, this is the main
	 * way to get yourself a listing object.
	 *
	 * @param int $listing_id
	 * @param bool $get_all If true, when fetching the listing, it gets all the
	 *  data for the listing all at once.  Handy to set to false if you know you
	 *  will only be using 1 or 2 columns so that it will be more efficient to only
	 *  retrieve those columns.
	 * @param bool $try_expired if true, will see if listing exists expired if it can't be found in active
	 * @param bool $force_refresh forces the system to get a fresh copy of the listing data, even if it's already been done this pageload.
	 *  Useful for places in the admin that update the listing directly but rely on this class for display after the fact
	 * @return geoListing|null If listing not found, will return null.
	 */
	public static function getListing($listing_id, $get_all = true, $try_expired = false, $force_refresh = false){
		$listing_id = intval($listing_id);
		if (!$listing_id){
			//invalid
			return null;
		}
		
		if($force_refresh && isset(self::$_listings[$listing_id])) {
			unset(self::$_listings[$listing_id]);
		}
		
		if (!isset(self::$_listings[$listing_id]) || (self::$_listings[$listing_id] === null && $try_expired)) {
			//listing has not been retrieved yet, or it has but is invalid but this time we are trying expired
			$db = DataAccess::getInstance();
			//get the listing and info
			self::$_listings[$listing_id] = new geoListing;
			self::$_listings[$listing_id]->_all = $get_all;
			self::$_listings[$listing_id]->_isExpired = false;
			
			if ($get_all){
				$sql = "SELECT * FROM ".geoTables::classifieds_table." WHERE `id`=? LIMIT 1";
				$listing_data = $db->GetRow($sql, array($listing_id));
				if (!$listing_data && $try_expired) {
					$sql = "SELECT * FROM ".geoTables::classifieds_expired_table." WHERE `id`=? LIMIT 1";
					$listing_data = $db->GetRow($sql, array($listing_id));
					self::$_listings[$listing_id]->_isExpired = true;
				}
				if ($listing_data) {
					//make sure ID is integer so strict comparisons work, since
					//the original way $listing->id worked was to be an int
					$listing_data['id'] = (int)$listing_data['id'];
					self::$_listings[$listing_id]->_listing_data = $listing_data;
				} else {
					//invalid listing!
					self::$_listings[$listing_id] = null;
				}
			} else {
				//see if it is valid
				
				$row = $db->GetRow("SELECT `id` FROM ".geoTables::classifieds_table." WHERE `id`=$listing_id LIMIT 1");
				self::$_listings[$listing_id]->_listing_data = array ('id' => $listing_id);
				
				if (!$row && $try_expired) {
					//see if it's in expired
					$row = $db->GetRow("SELECT `id` FROM ".geoTables::classifieds_expired_table." WHERE `id`=$listing_id LIMIT 1");
					self::$_listings[$listing_id]->_isExpired = true;
				}
				if (!is_array($row) || !isset($row['id']) || $row['id'] != $listing_id) {
					//invalid listing
					self::$_listings[$listing_id] = null;
				}
			}
		}
		
		if (!$try_expired && isset(self::$_listings[$listing_id]) && self::$_listings[$listing_id]->_isExpired) {
			//it exists but it is expired, and we said no to getting expired
			return null;
		}
		
		if (is_object(self::$_listings[$listing_id]) && $get_all && !self::$_listings[$listing_id]->_all) {
			//get all the additional data about the listing, previously we didn't get all the data
			$db = DataAccess::getInstance();
			
			$table = (self::$_listings[$listing_id]->_isExpired)? geoTables::classifieds_expired_table: geoTables::classifieds_table;
			self::$_listings[$listing_id]->_listing_data = $db->GetRow("SELECT * FROM $table WHERE `id`=$listing_id LIMIT 1");
			self::$_listings[$listing_id]->_all = true;
		}
		
		return self::$_listings[$listing_id];
	}
	
	/**
	 * Whether or not this listing is from the expired table or not.
	 * @return bool
	 */
	public function isExpired ()
	{
		return $this->_isExpired;
	}
	
	/**
	 * Finds out if this listing is locked for editing
	 *
	 * @return int 1 if locked, 0 otherwise
	 */	
	public function isLocked ()
	{
		$originalOrderItemID = $this->order_item_id;
		$orderItem = geoOrderItem::getOrderItem($originalOrderItemID);
		if (is_object($orderItem)) {
			return $orderItem->get('locked', 0);
		} 
		return 0;
	}
	
	/**
	 * Locks a listing, preventing other processes from modifying it until the lock is released
	 * Also can reverse that process by passing it 0 or false
	 *
	 * @param boolean|int $state true|1 to lock, false|0 to unlock 
	 * @return boolean true on success, false on failure
	 */
	public function setLocked ($state = true)
	{
		//handle boolean values passed to function
		$state = (($state) ? 1 : false);
		
		$originalOrderItemID = $this->order_item_id;
		$orderItem = geoOrderItem::getOrderItem($originalOrderItemID);
		if (is_object($orderItem)) {
			$orderItem->set('locked', $state);
			$orderItem->save();
			return true;
		}
		return false;
	}
	
	/**
	 * Converts the listing's data into an array and returns it.
	 *
	 * @return array
	 */
	public function toArray ()
	{
		return $this->_listing_data;
	}
	
	/**
	 * Gets an array of all the order item ID's for parent items that have listing or listing_id set to
	 * this listing's ID.  Note that neither input vars will apply to the "original" order item for the
	 * listing.
	 * 
	 * This does NOT verify order items.  It will also not get any order items if
	 * $listing->order_item_id is not set.
	 * 
	 * @param bool $onlyActive If true, will only return order items current "active"
	 * @param bool $onlyLegit If true(default), will only return order items with the Order field set.
	 * @return array An array of order item ID's, with the first one being the "original"
	 *  order item (set as $listing->order_item_id).
	 */
	public function getAllOrderItems ($onlyActive = false, $onlyLegit = true)
	{
		if (!$this->id || !$this->order_item_id) {
			//no order items to get
			return array();
		}
		$extra = '';
		if ($onlyActive) {
			$extra .= " item.status = 'active' AND ";
		}
		if ($onlyLegit) {
			$extra .= " item.order != 0 AND "; 
		}
		//Get all the items that go with this listing ID for "legit" items (that have orders)
		$sql = "SELECT item.id as item_id FROM `geodesic_order_item` as item, `geodesic_order_item_registry` as regi 
		WHERE regi.order_item = item.id AND item.parent='0' AND $extra 
		 (regi.index_key='listing_id' OR regi.index_key='listing')
		 AND regi.val_string = ?
		ORDER BY item.id";
		
		$all_items = DataAccess::getInstance()->GetAll($sql, array($this->id));
		if (!isset($all_items[0]) || $all_items[0]['item_id'] != $this->order_item_id) {
			//item ID is not first in list, how did that happen?  Most likely it is because it is not
			//active.  
			
			//We are going to force the first order item to always be used as the starter when getting order_items.
			array_unshift($all_items, array('item_id' => $this->order_item_id));
		}
		//put them in an array
		$return = array();
		foreach ($all_items as $row) {
			if (!in_array($row['item_id'], $return)) {
				$return[] = $row['item_id'];
			}
		}
		return $return;
	}
	
	/**
	 * If you get a set of listing's data, you can use this method to populate
	 * listing objects for each of the retrieved listings.
	 * 
	 * That way it does not have to get info for the same listing twice.
	 * 
	 * @param array $dataSet in array form, as if $db->GetAll($sql) was used.
	 * @since Version 5.0.0
	 */
	public static function addDataSet ($dataSet)
	{
		if (!is_array($dataSet)) {
			//expect data to be an array of listing info
			return;
		}
		foreach ($dataSet as $row) {
			self::addListingData($row);
		}
	}
	
	/**
	 * Add single listing's data so it doesn't have to be retrieved later.
	 * @param array $listing
	 * @since Version 5.1.2
	 */
	public static function addListingData ($listing)
	{
		if (!is_array($listing)) {
			//expect data to be an array of listing info
			return;
		}
		if (!isset($listing['id'])) {
			//no ID set? can't do anything without ID's
			return;
		}
		$listing_id = $listing['id'] = (int)$listing['id'];
		if (!$listing_id) {
			//invalid data set?
			return;
		}
		if (!isset(self::$_listings[$listing_id]) || self::$_listings[$listing_id] === null) {
			self::$_listings[$listing_id] = new geoListing;
			self::$_listings[$listing_id]->_all = false;//assume it is not all the data
			self::$_listings[$listing_id]->_isExpired = false;//assume it is not expired
			self::$_listings[$listing_id]->_listing_data = $listing;
		} else {
			if (!is_array(self::$_listings[$listing_id]->_listing_data)) {
				self::$_listings[$listing_id]->_listing_data = array();
			}
			self::$_listings[$listing_id]->_listing_data = array_merge(self::$_listings[$listing_id]->_listing_data, $listing);
		}
	}
	
	/**
	 * Remove the specified listing.  This is meant to be used in a mass-deletion
	 * so the category count is not updated, and removes it directly, not move
	 * it to the archive table.  All attached images, questions, and bids are
	 * also removed.
	 * 
	 * @param int $listingId
	 * @param bool $isArchived If true, will behave as it should when a listing
	 *   is merely being archived.  Note that it still does NOT add the listing
	 *   to the archived table, that should be done prior to calling this method.
	 *   This param added in version 5.2.0.
	 * @return bool Whether removal was a success or not.
	 * @since Version 5.0.0
	 */
	public static function remove ($listingId, $isArchived = false)
	{
		$listingId = (int)$listingId;
		
		if (!$listingId) {
			//can't remove without a proper listing ID
			return false;
		}
		$db = DataAccess::getInstance();

		//delete url images
		$sql = "SELECT `image_id` FROM ".geoTables::images_urls_table." WHERE `classified_id`=?";
		$get_url_result = $db->Execute($sql, array($listingId));
		if (!$get_url_result) {
			trigger_error('ERROR SQL: Error getting images, using sql: '.$sql.', Error: '.$db->ErrorMsg());
			return false;
		}
		while ($row = $get_url_result->FetchRow()) {
			//using this way as it takes less resources than something like FetchAll
			$imageId = (int)$row['image_id'];
			if ($imageId) {
				$removeImage = geoImage::remove($imageId);
				if (!$removeImage) {
					trigger_error('ERROR STATS: Stopping removal of listing, could not remove image(s).');
					return false;
				}
			}
		}
		
		//remove "simple" things from listings that don't need anything more
		//than removing entries from the DB
		$simpleRemoves = array (
			geoTables::classified_extra_table => 'classified_id',//extra questions
			geoTables::bid_table => 'auction_id',//bids
			geoTables::autobid_table => 'auction_id',//auto-bids
			geoTables::voting_table => 'classified_id',//voting
			geoTables::tags => 'listing_id',//tags
			geoTables::offsite_videos => 'listing_id',//offsite videos
		);
		if (!$isArchived) {
			//also remove from feedback tables
			$simpleRemoves[geoTables::auctions_feedbacks_table] = 'auction_id';//feedback
		}
		foreach ($simpleRemoves as $tableName => $colName) {
			//delete all entries that match this listing
			$sql = "DELETE FROM {$tableName} WHERE `{$colName}`={$listingId}";
			$result = $db->Execute($sql);
			if (!$result) {
				trigger_error('ERROR SQL: Error removing stuff from listing, using sql: '.$sql.', Error: '.$db->ErrorMsg());
				return false;
			}
		}
		
		//just in case there are any already retrieved listing data...
		if (isset(self::$_listings[$categoryId])) {
			//reset the listing data so there is nothing to go on to update things
			//in case something already has a reference to it.
			self::$_listings[$listingId]->_listing_data = array();
			
			//unset so it can't be requested later
			unset(self::$_listings[$listingId]);
		}
		
		//call for addons to do something when listing is removed...
		geoAddon::triggerUpdate('notify_geoListing_remove', array('listingId' => $listingId, 'isArchived' => $isArchived));
		
		//delete from classifieds table
		$sql = "DELETE FROM ".geoTables::classifieds_table." WHERE `id` = {$listingId}";
		$remove_result = $db->Execute($sql);
		if (!$remove_result) {
			trigger_error('ERROR SQL: Error removing listing, using sql: '.$sql.', Error: '.$db->ErrorMsg());
			return false;
		}
		return true;
	}
	
	/**
	 * Allows object oriented listing objects.
	 *
	 * @param string $name
	 * @return mixed
	 */
	public function __get($name){
		$name = strtolower($name);
		if (!$this->_all && !isset($this->_listing_data[$name])){
			$db = DataAccess::getInstance();
			$table = ($this->_isExpired)? geoTables::classifieds_expired_table : geoTables::classifieds_table;
			$sql = "SELECT `$name` FROM $table WHERE `id`=? LIMIT 1";
			$row = $db->GetRow($sql, array($this->id));
			$this->_listing_data[$name] = $row[$name];
		}
		
		if (isset($this->_listing_data[$name])){
			return $this->_listing_data[$name];
		}
		return false;
	}
	
	/**
	 * Allows object oriented listing objects.  (not meant to be called directly)
	 * 
	 * Allows you to use $listing->price = 1.11 to set the price on a listing.
	 *
	 * @param string $name
	 * @param string $value
	 */
	public function __set($name, $value){
		if ($this->_isExpired && $name != 'order_item_id') {
			//if expired, do NOT allow setting values (unless it's order_item_id)
			return false;
		}
		
		if (isset($this->_listing_data[$name]) && $this->_listing_data[$name] == $value) {
			//nothing to change, it's already set to this.
			return true;
		}
		$this->_listing_data[$name] = $value;
		
		$db = DataAccess::getInstance();
		$table = ($this->_isExpired)? geoTables::classifieds_expired_table : geoTables::classifieds_table;
		$sql = "UPDATE $table SET `$name`=?  WHERE `id`=? LIMIT 1";
		$result = $db->Execute($sql, array($value, $this->_listing_data['id']));
		if ($result) {
			return true;
		}

		return false;
	}
}